<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login');
    exit;
}
include '../includes/header.php';
include '../config/database.php';

$statusMessage = '';
if (isset($_GET['status'])) {
    if ($_GET['status'] === 'submitted') {
        $statusMessage = 'Your details have been submitted.';
    } elseif ($_GET['status'] === 'saved') {
        $statusMessage = 'Your details have been saved. You can continue later.';
    } elseif ($_GET['status'] === 'error_upload') {
        $statusMessage = 'There was an error uploading your file. Please try again.';
    } elseif ($_GET['status'] === 'invalid_file_type') {
        $statusMessage = 'Invalid file type. Only PDF and image files (JPG, PNG) are allowed.';
    }
}

// Fetch the logged-in user's group and existing details if any
$userId = $_SESSION['user_id'];
$stmt = $pdo->prepare('SELECT `group`, (SELECT status FROM student_details WHERE user_id = ?) AS submission_status FROM users WHERE id = ?');
$stmt->execute([$userId, $userId]);
$user = $stmt->fetch();
$group = htmlspecialchars($user['group'], ENT_QUOTES, 'UTF-8');
$submissionStatus = htmlspecialchars($user['submission_status'], ENT_QUOTES, 'UTF-8');

$isSubmitted = $submissionStatus === 'submitted';
$isSaved = $submissionStatus === 'saved';

if ($isSaved || $isSubmitted) {
    $stmt = $pdo->prepare('SELECT * FROM student_details WHERE user_id = ?');
    $stmt->execute([$userId]);
    $studentDetails = $stmt->fetch(PDO::FETCH_ASSOC);
} else {
    $studentDetails = [
        'surname' => '',
        'first_name' => '',
        'other_names' => '',
        'telephone' => '',
        'email' => '',
        'course' => '',
        'organisation_name' => '',
        'months' => '',
        'location' => '',
        'gps_location' => '',
        'region' => '',
        'district' => '',
        'acceptance_letter' => ''
    ];
}

// Fetch regions from the database
$stmt = $pdo->prepare('SELECT DISTINCT region FROM districts');
$stmt->execute();
$regions = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Define courses of study based on the group
$courses = [];
if ($group === 'Industrial Internship') {
    $courses = [
        'Accounting',
        'Procurement and Supply Chain Management',
        'Business Information Systems',
        'Human Resource Management',
        'Banking and Finance',
        'Insurance and Risk Management',
        'Marketing and Entrepreneurship'
    ];
} elseif ($group === 'Teaching Practice') {
    $courses = [
        'Management Education',
        'Accounting Education'
    ];
}
?>

<div class="container mt-4">
    <?php if ($statusMessage): ?>
        <div class="alert alert-info" id="statusMessage">
            <?php echo htmlspecialchars($statusMessage, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <style>
        .btn-custom {
            background-color: #007bff;
            color: white !important;
            border: none;
            border-radius: 0px;
            padding: 10px 20px;
            text-align: center;
            text-decoration: none;
            display: inline-block;
            font-size: 16px;
            margin: 4px 2px;
            cursor: pointer;
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
            transition: background-color 0.3s, box-shadow 0.3s;
        }

        .btn-custom:hover {
            background-color: #0056b3;
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }

        .spinner {
            display: none;
            margin-left: 10px;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #007bff;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>

    <div class="row justify-content-center">
        <div class="col-md-8 col-lg-6">
            <h2>University of Education, Winneba</h2>
            <h4>School of Business.</h4>

            <?php if ($isSubmitted): ?>
                <div class="card mb-4">
                    <div class="card-header">
                        <h3>Download Supervisor's Evaluation Form</h3>
                    </div>
                    <div class="card-body text-center">
                        <a href="download_evaluation" class="btn btn-custom">
                            <i class="fas fa-download btn-icon"></i>Download
                        </a>
                    </div>
                </div>
            <?php else: ?>
                <div class="card mb-4">
                    <div class="card-header">
                        <h3>Student's Details</h3>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="save_student_details" enctype="multipart/form-data" id="studentForm">
                            <fieldset <?php echo $isSubmitted ? 'disabled' : ''; ?>>
                                <div class="mb-3">
                                    <label for="surname" class="form-label">Surname</label>
                                    <input type="text" class="form-control" id="surname" name="surname" value="<?php echo htmlspecialchars($studentDetails['surname'], ENT_QUOTES, 'UTF-8'); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="firstName" class="form-label">First Name</label>
                                    <input type="text" class="form-control" id="firstName" name="firstName" value="<?php echo htmlspecialchars($studentDetails['first_name'], ENT_QUOTES, 'UTF-8'); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="otherNames" class="form-label">Other Names</label>
                                    <input type="text" class="form-control" id="otherNames" name="otherNames" value="<?php echo htmlspecialchars($studentDetails['other_names'], ENT_QUOTES, 'UTF-8'); ?>">
                                </div>
                                <div class="mb-3">
                                    <label for="telephone" class="form-label">Telephone Number</label>
                                    <input type="tel" class="form-control" id="telephone" name="telephone" pattern="[0-9]{10,}" value="<?php echo htmlspecialchars($studentDetails['telephone'], ENT_QUOTES, 'UTF-8'); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="email" class="form-label">Email</label>
                                    <input type="email" class="form-control" id="email" name="email" value="<?php echo htmlspecialchars($studentDetails['email'], ENT_QUOTES, 'UTF-8'); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="course" class="form-label">Course of Study</label>
                                    <select class="form-control" id="course" name="course" required>
                                        <option value="">Select Course</option>
                                        <?php foreach ($courses as $course): ?>
                                            <option value="<?php echo htmlspecialchars($course, ENT_QUOTES, 'UTF-8'); ?>" <?php echo htmlspecialchars($studentDetails['course'], ENT_QUOTES, 'UTF-8') === $course ? 'selected' : ''; ?>><?php echo htmlspecialchars($course, ENT_QUOTES, 'UTF-8'); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </fieldset>
                    </div>
                </div>

                <div class="card mb-4">
                    <div class="card-header">
                        <h3><?php echo $group === 'Industrial Internship' ? 'Organisation' : 'School'; ?></h3>
                    </div>
                    <div class="card-body">
                            <fieldset <?php echo $isSubmitted ? 'disabled' : ''; ?>>
                                <div class="mb-3">
                                    <label for="organisationName" class="form-label"><?php echo $group === 'Industrial Internship' ? 'Name of Organisation' : 'Name of School'; ?></label>
                                    <input type="text" class="form-control" id="organisationName" name="organisationName" value="<?php echo htmlspecialchars($studentDetails['organisation_name'], ENT_QUOTES, 'UTF-8'); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="months" class="form-label">Number of Months</label>
                                    <select class="form-control" id="months" name="months" required>
                                        <option value="">Select Months</option>
                                        <?php for ($i = 1; $i <= 6; $i++): ?>
                                            <option value="<?php echo htmlspecialchars($i, ENT_QUOTES, 'UTF-8'); ?>" <?php echo htmlspecialchars($studentDetails['months'], ENT_QUOTES, 'UTF-8') == $i ? 'selected' : ''; ?>><?php echo htmlspecialchars($i, ENT_QUOTES, 'UTF-8'); ?> month<?php echo $i > 1 ? 's' : ''; ?></option>
                                        <?php endfor; ?>
                                    </select>
                                </div>
                                <div class="mb-3">
                                    <label for="location" class="form-label"><?php echo $group === 'Industrial Internship' ? 'Location of Organisation' : 'Location of School'; ?></label>
                                    <input type="text" class="form-control" id="location" name="location" value="<?php echo htmlspecialchars($studentDetails['location'], ENT_QUOTES, 'UTF-8'); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="gpsLocation" class="form-label">
                                        Ghana Post GPS / 
                                        <a href="https://www.google.com/maps/" target="_blank">Google location </a> of the Organisation
                                    </label>
                                    <input type="text" class="form-control" id="gpsLocation" name="gpsLocation" value="<?php echo htmlspecialchars($studentDetails['gps_location'], ENT_QUOTES, 'UTF-8'); ?>" required>
                                    <small id="gpsError" class="form-text text-danger" style="display:none;">Please enter a valid GPS location (e.g., coordinate, Google Maps URL, or Ghana Post GPS code).</small>
                                </div>
                                <div class="mb-3">
                                    <label for="region" class="form-label">Region</label>
                                    <select class="form-control" id="region" name="region" required>
                                        <option value="">Select Region</option>
                                        <?php foreach ($regions as $region): ?>
                                            <option value="<?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo htmlspecialchars($studentDetails['region'], ENT_QUOTES, 'UTF-8') === $region['region'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="mb-3">
                                    <label for="district" class="form-label">District</label>
                                    <select class="form-control" id="district" name="district" required>
                                        <option value="">Select District</option>
                                        <!-- Fill districts based on selected region -->
                                        <?php if ($studentDetails['region']): ?>
                                            <?php
                                            $stmt = $pdo->prepare('SELECT district FROM districts WHERE region = ?');
                                            $stmt->execute([htmlspecialchars($studentDetails['region'], ENT_QUOTES, 'UTF-8')]);
                                            $districts = $stmt->fetchAll(PDO::FETCH_ASSOC);
                                            foreach ($districts as $district):
                                            ?>
                                                <option value="<?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo htmlspecialchars($studentDetails['district'], ENT_QUOTES, 'UTF-8') === $district['district'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?></option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                </div>
                                <div class="mb-3">
                                    <label for="acceptanceLetter" class="form-label">Upload Acceptance Letter</label>
                                    <input type="file" class="form-control" id="acceptanceLetter" name="acceptanceLetter" accept=".pdf, .jpg, .jpeg, .png" required>
                                    <small id="fileError" class="form-text text-danger" style="display:none;">Please upload a valid acceptance letter.</small>
                                    <div class="spinner" id="processingSpinner"></div>
                                    <?php if ($studentDetails['acceptance_letter']): ?>
                                        <p>Current file: <?php echo htmlspecialchars(basename($studentDetails['acceptance_letter']), ENT_QUOTES, 'UTF-8'); ?></p>
                                    <?php endif; ?>
                                </div>

                                <?php if (!$isSubmitted): ?>
                                <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                    <!--button type="submit" name="save" class="btn btn-secondary" style="border-radius: 0px;">Save and Continue Later</button-->
                                    <button type="submit" name="submit" class="btn btn-primary" style="border-radius: 0px;" id="submitBtn">Submit</button>
                                </div>
                                <?php endif; ?>
                            </fieldset>
                        </form>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Add PDF.js and Tesseract.js for file scanning -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdf.js/2.10.377/pdf.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/tesseract.js@v4.0.2/dist/tesseract.min.js"></script>

<script>
    // Fetch districts based on the selected region
    document.getElementById('region').addEventListener('change', function () {
        const region = this.value;
        const districtSelect = document.getElementById('district');
        districtSelect.innerHTML = '<option value="">Select District</option>';
        if (region) {
            fetch('fetch_districts?region=' + encodeURIComponent(region))
                .then(response => response.json())
                .then(data => {
                    data.forEach(district => {
                        const option = document.createElement('option');
                        option.value = district.district;
                        option.textContent = district.district;
                        districtSelect.appendChild(option);
                    });
                });
        }
    });

    document.addEventListener('DOMContentLoaded', function () {
        const gpsInput = document.getElementById('gpsLocation');
        const gpsError = document.getElementById('gpsError');
        const form = document.getElementById('studentForm');
        const submitBtn = document.getElementById('submitBtn');
        let isSubmitAttempted = false; // Tracks if "Submit" button was clicked

        // Function to validate GPS location
        function validateGPS() {
            const gpsValue = gpsInput.value.trim();
            const coordinateRegex = /^-?([1-8]?\d(\.\d+)?|90(\.0+)?),\s*-?(180(\.0+)?|((1[0-7]\d)|([1-9]?\d))(\.\d+)?)$/;
            const googleMapsUrlRegex = /^https:\/\/(maps\.app\.goo\.gl\/[A-Za-z0-9]+|www\.google\.com\/maps\/.*)$/;
            const ghanaPostRegex = /^[A-Za-z0-9+-]{5,}(\s*,?\s*[A-Za-z\s]+)?$/;

            if (coordinateRegex.test(gpsValue) || googleMapsUrlRegex.test(gpsValue) || ghanaPostRegex.test(gpsValue)) {
                gpsError.style.display = 'none';
                return true;
            } else {
                gpsError.style.display = 'block';
                return false;
            }
        }

        // Validate both GPS location and acceptance letter before submission
        function validateForm() {
            const isGPSValid = validateGPS();
            const isFileValid = fileIsValid;

            return isGPSValid && isFileValid;
        }

        // Real-time GPS validation
        gpsInput.addEventListener('input', validateGPS);

        // Track if "Submit" button is clicked
        submitBtn.addEventListener('click', function () {
            isSubmitAttempted = true;  // Set the flag to indicate "Submit" button was clicked
        });
    });
</script>

<?php include '../includes/footer.php'; ?>
